<?php
/**
 * Plugin Name: MayWeb License API
 * Description: REST API for MayWeb license checks & trials.
 * Version: 1.2.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class Mayweb_License_API {
    // Device activation table (per-device records)
    const TABLE_ACTIVATIONS = 'mayweb_licenses';

    // Master license keys table
    const TABLE_LICENSES = 'mayweb_license_keys';

    // !!! CHANGE THIS SECRET TO YOUR OWN RANDOM STRING !!!
    const SECRET_SALT = 'CHANGE_THIS_TO_A_LONG_RANDOM_SECRET_2025';

    public function __construct() {
        register_activation_hook(__FILE__, [$this, 'install']);
        add_action('rest_api_init', [$this, 'register_routes']);
    }

    public function install() {
        global $wpdb;
        $charset_collate = $wpdb->get_charset_collate();

        $table_activations = $wpdb->prefix . self::TABLE_ACTIVATIONS;
        $table_licenses    = $wpdb->prefix . self::TABLE_LICENSES;

        require_once ABSPATH . 'wp-admin/includes/upgrade.php';

        // Per-device activations (very similar to your old table)
        $sql_activations = "CREATE TABLE $table_activations (
            id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT PRIMARY KEY,
            license_key VARCHAR(64) NULL,
            device_id VARCHAR(191) NOT NULL,
            device_hash CHAR(64) NOT NULL,
            plan VARCHAR(32) NOT NULL DEFAULT 'TRIAL',        -- TRIAL / PRO / FREE
            status VARCHAR(32) NOT NULL DEFAULT 'active',     -- active / expired / banned / disabled
            max_devices INT UNSIGNED NOT NULL DEFAULT 1,
            used_devices INT UNSIGNED NOT NULL DEFAULT 1,
            trial_expires_at DATETIME NULL,
            expires_at DATETIME NULL,
            ip_address VARCHAR(45) NULL,
            ip_prefix VARCHAR(32) NULL,
            user_agent TEXT NULL,
            last_check_at DATETIME NULL,
            app_version VARCHAR(50) DEFAULT NULL,
            created_at DATETIME NOT NULL,
            updated_at DATETIME NOT NULL,
            UNIQUE KEY uniq_license_device (license_key, device_hash),
            KEY idx_device_hash (device_hash),
            KEY idx_ip_prefix (ip_prefix),
            KEY idx_license_key (license_key)
        ) $charset_collate;";

        // Master license keys table (one row per license sold/generated)
        $sql_licenses = "CREATE TABLE $table_licenses (
            id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT PRIMARY KEY,
            license_key VARCHAR(64) NOT NULL,
            plan VARCHAR(32) NOT NULL DEFAULT 'PRO',          -- PRO / TRIAL / FREE
            status VARCHAR(32) NOT NULL DEFAULT 'active',     -- active / expired / banned / disabled
            max_devices INT UNSIGNED NOT NULL DEFAULT 1,
            expires_at DATETIME NULL,
            notes TEXT NULL,
            created_at DATETIME NOT NULL,
            updated_at DATETIME NOT NULL,
            UNIQUE KEY uniq_license_key (license_key),
            KEY idx_status (status)
        ) $charset_collate;";

        dbDelta($sql_activations);
        dbDelta($sql_licenses);
    }

    public function register_routes() {
        // Check license / trial status
        register_rest_route(
            'mayweb/v1',
            '/check',
            [
                'methods'  => 'POST',
                'callback' => [$this, 'handle_check'],
                'permission_callback' => '__return_true',
            ]
        );

        // Start trial for a device
        register_rest_route(
            'mayweb/v1',
            '/start-trial',
            [
                'methods'  => 'POST',
                'callback' => [$this, 'handle_start_trial'],
                'permission_callback' => '__return_true',
            ]
        );

        // Admin-only: generate license keys
        register_rest_route(
            'mayweb/v1',
            '/licenses',
            [
                'methods'  => 'POST',
                'callback' => [$this, 'handle_create_license'],
                'permission_callback' => function () {
                    return current_user_can('manage_options');
                },
            ]
        );
    }

    /* ----------------- Helpers ----------------- */

    protected function get_client_ip() {
        $keys = [
            'HTTP_X_FORWARDED_FOR',
            'HTTP_CLIENT_IP',
            'HTTP_CF_CONNECTING_IP',
            'REMOTE_ADDR',
        ];

        foreach ($keys as $key) {
            if (!empty($_SERVER[$key])) {
                $value = $_SERVER[$key];
                $parts = explode(',', $value);
                $ip = trim($parts[0]);
                if (filter_var($ip, FILTER_VALIDATE_IP)) {
                    return $ip;
                }
            }
        }
        return '0.0.0.0';
    }

    protected function get_ip_prefix($ip) {
        if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4)) {
            $parts = explode('.', $ip);
            if (count($parts) >= 3) {
                return $parts[0] . '.' . $parts[1] . '.' . $parts[2];
            }
            return $ip;
        }

        if (strpos($ip, ':') !== false) {
            $parts = explode(':', $ip);
            return implode(':', array_slice($parts, 0, 4));
        }

        return $ip;
    }

    protected function make_device_hash($device_id, $user_agent) {
        $ua = substr((string) $user_agent, 0, 160);
        return hash('sha256', self::SECRET_SALT . '|' . $device_id . '|' . $ua);
    }

    protected function now() {
        return current_time('mysql');
    }

    protected function future_datetime($hours) {
        $t = time() + ($hours * 3600);
        return gmdate('Y-m-d H:i:s', $t);
    }

    protected function mask_license($license_key) {
        if (!$license_key) return null;
        $len = strlen($license_key);
        if ($len <= 8) return $license_key;
        return substr($license_key, 0, 4) . str_repeat('*', max(0, $len - 8)) . substr($license_key, -4);
    }

    protected function is_expired($datetime) {
        if (!$datetime) return false;
        $ts = strtotime($datetime);
        if (!$ts) return false;
        return $ts < time();
    }

    protected function feature_flags_for_plan($plan) {
        $plan = strtoupper($plan);
        $features = [
            'canUseSftpDeploy'   => false,
            'canUseCpanelDeploy' => false,
        ];

        switch ($plan) {
            case 'PRO':
                $features['canUseSftpDeploy']   = true;
                $features['canUseCpanelDeploy'] = true;
                break;
            case 'TRIAL':
                $features['canUseSftpDeploy']   = true;
                $features['canUseCpanelDeploy'] = true;
                break;
            case 'FREE':
            default:
                // Keep false for now
                break;
        }

        return $features;
    }

    protected function generate_license_key($length = 32) {
        // 32 hex chars = 16 bytes
        $bytes = random_bytes((int) ($length / 2));
        return strtoupper(bin2hex($bytes));
    }

    /* ----------------- /check ----------------- */

    public function handle_check(WP_REST_Request $request) {
        global $wpdb;
        $table_act = $wpdb->prefix . self::TABLE_ACTIVATIONS;
        $table_lic = $wpdb->prefix . self::TABLE_LICENSES;

        $params = $request->get_json_params();
        $device_id   = isset($params['device_id']) ? sanitize_text_field($params['device_id']) : '';
        $license_key = isset($params['license_key']) ? sanitize_text_field($params['license_key']) : '';
        $app_version = isset($params['app_version']) ? sanitize_text_field($params['app_version']) : '';

        if ($device_id === '') {
            return new WP_REST_Response([
                'status'  => 'error',
                'message' => 'Missing device_id.',
            ], 400);
        }

        $ip         = $this->get_client_ip();
        $ip_prefix  = $this->get_ip_prefix($ip);
        $user_agent = isset($_SERVER['HTTP_USER_AGENT']) ? substr(wp_unslash($_SERVER['HTTP_USER_AGENT']), 0, 255) : '';

        $device_hash = $this->make_device_hash($device_id, $user_agent);
        $now         = $this->now();

        // 1) License key flow (PRO/TRIAL/FREE via master table)
        if ($license_key !== '') {
            // Look up master license key
            $license = $wpdb->get_row(
                $wpdb->prepare(
                    "SELECT * FROM $table_lic WHERE license_key = %s LIMIT 1",
                    $license_key
                )
            );

            if (!$license) {
                return new WP_REST_Response([
                    'status'  => 'error',
                    'plan'    => 'FREE',
                    'message' => 'Invalid license key.',
                ], 403);
            }

            $plan   = $license->plan ?: 'PRO';
            $status = $license->status ?: 'active';

            // Check if license itself is expired
            if ($license->expires_at && $this->is_expired($license->expires_at)) {
                $status = 'expired';
                $wpdb->update(
                    $table_lic,
                    [
                        'status'     => 'expired',
                        'updated_at' => $now,
                    ],
                    ['id' => $license->id],
                    ['%s','%s'],
                    ['%d']
                );
            }

            if ($status === 'banned' || $status === 'disabled') {
                return new WP_REST_Response([
                    'status'  => 'error',
                    'plan'    => $plan,
                    'message' => 'License disabled. Contact support.',
                ], 403);
            }

            if ($status === 'expired') {
                return new WP_REST_Response([
                    'status'      => 'expired',
                    'plan'        => $plan,
                    'message'     => 'Your license has expired.',
                    'license_key' => $this->mask_license($license_key),
                    'features'    => $this->feature_flags_for_plan('FREE'),
                    'expires_at'  => $license->expires_at,
                    'app_version' => $app_version,
                ], 200);
            }

            // Count distinct devices already using this key
            $used_devices = (int) $wpdb->get_var(
                $wpdb->prepare(
                    "SELECT COUNT(DISTINCT device_hash) FROM $table_act WHERE license_key = %s",
                    $license_key
                )
            );
            $max_devices  = max(1, (int) $license->max_devices);

            if ($used_devices >= $max_devices) {
                // Check if this exact device already has activation; if yes, allow it
                $existing_activation = $wpdb->get_row(
                    $wpdb->prepare(
                        "SELECT * FROM $table_act WHERE license_key = %s AND device_hash = %s LIMIT 1",
                        $license_key,
                        $device_hash
                    )
                );

                if (!$existing_activation) {
                    return new WP_REST_Response([
                        'status'  => 'error',
                        'plan'    => $plan,
                        'message' => 'This license key has reached its device limit.',
                    ], 403);
                }

                // Device is already registered, refresh info and return
                $wpdb->update(
                    $table_act,
                    [
                        'ip_address'    => $ip,
                        'ip_prefix'     => $ip_prefix,
                        'user_agent'    => $user_agent,
                        'last_check_at' => $now,
                        'app_version'   => $app_version,
                        'updated_at'    => $now,
                    ],
                    ['id' => $existing_activation->id],
                    ['%s','%s','%s','%s','%s','%s'],
                    ['%d']
                );

                $features = $this->feature_flags_for_plan($plan);

                return new WP_REST_Response([
                    'status'        => $existing_activation->status,
                    'plan'          => $plan,
                    'message'       => 'License active on this device.',
                    'license_key'   => $this->mask_license($license_key),
                    'features'      => $features,
                    'expires_at'    => $license->expires_at,
                    'trial_expires_at' => null,
                    'app_version'   => $app_version,
                ], 200);
            }

            // Under device limit: either create or update activation
            $row = $wpdb->get_row(
                $wpdb->prepare(
                    "SELECT * FROM $table_act WHERE license_key = %s AND device_hash = %s LIMIT 1",
                    $license_key,
                    $device_hash
                )
            );

            if (!$row) {
                $wpdb->insert(
                    $table_act,
                    [
                        'license_key'      => $license_key,
                        'device_id'        => $device_id,
                        'device_hash'      => $device_hash,
                        'plan'             => $plan,
                        'status'           => 'active',
                        'max_devices'      => $max_devices,
                        'used_devices'     => $used_devices + 1,
                        'trial_expires_at' => null,
                        'expires_at'       => $license->expires_at,
                        'ip_address'       => $ip,
                        'ip_prefix'        => $ip_prefix,
                        'user_agent'       => $user_agent,
                        'last_check_at'    => $now,
                        'app_version'      => $app_version,
                        'created_at'       => $now,
                        'updated_at'       => $now,
                    ]
                );

                $row = $wpdb->get_row(
                    $wpdb->prepare(
                        "SELECT * FROM $table_act WHERE id = %d",
                        $wpdb->insert_id
                    )
                );
            } else {
                $wpdb->update(
                    $table_act,
                    [
                        'ip_address'    => $ip,
                        'ip_prefix'     => $ip_prefix,
                        'user_agent'    => $user_agent,
                        'last_check_at' => $now,
                        'app_version'   => $app_version,
                        'updated_at'    => $now,
                    ],
                    ['id' => $row->id],
                    ['%s','%s','%s','%s','%s','%s'],
                    ['%d']
                );
            }

            $features = $this->feature_flags_for_plan($plan);

            return new WP_REST_Response([
                'status'        => $row->status,
                'plan'          => $plan,
                'message'       => 'License active.',
                'license_key'   => $this->mask_license($license_key),
                'features'      => $features,
                'expires_at'    => $license->expires_at,
                'trial_expires_at' => $row->trial_expires_at,
                'app_version'   => $app_version,
            ], 200);
        }

        // 2) No license key => Trial / FREE based on device_hash only (activations table)
        $row = $wpdb->get_row(
            $wpdb->prepare(
                "SELECT * FROM $table_act WHERE device_hash = %s ORDER BY id DESC LIMIT 1",
                $device_hash
            )
        );

        if (!$row) {
            return new WP_REST_Response([
                'status'   => 'inactive',
                'plan'     => 'FREE',
                'message'  => 'No license or trial found for this device.',
                'features' => $this->feature_flags_for_plan('FREE'),
            ], 200);
        }

        $wpdb->update(
            $table_act,
            [
                'ip_address'    => $ip,
                'ip_prefix'     => $ip_prefix,
                'user_agent'    => $user_agent,
                'last_check_at' => $now,
                'app_version'   => $app_version,
                'updated_at'    => $now,
            ],
            ['id' => $row->id],
            ['%s','%s','%s','%s','%s','%s'],
            ['%d']
        );

        $plan   = $row->plan ?: 'TRIAL';
        $status = $row->status ?: 'active';

        if ($status === 'banned' || $status === 'disabled') {
            return new WP_REST_Response([
                'status'   => 'error',
                'plan'     => $plan,
                'message'  => 'This device is blocked. Contact support.',
                'features' => $this->feature_flags_for_plan('FREE'),
            ], 403);
        }

        if ($plan === 'TRIAL') {
            if ($row->trial_expires_at && $this->is_expired($row->trial_expires_at)) {
                $status = 'expired';
                $wpdb->update(
                    $table_act,
                    [
                        'status'     => 'expired',
                        'updated_at' => $now,
                    ],
                    ['id' => $row->id],
                    ['%s','%s'],
                    ['%d']
                );
            }
        }

        $features = $this->feature_flags_for_plan($plan);
        if ($status !== 'active') {
            $features = $this->feature_flags_for_plan('FREE');
        }

        return new WP_REST_Response([
            'status'          => $status,
            'plan'            => $plan,
            'message'         => $status === 'expired'
                                    ? 'Your trial has expired.'
                                    : 'Trial active.',
            'license_key'     => $this->mask_license($row->license_key),
            'features'        => $features,
            'trial_expires_at'=> $row->trial_expires_at,
            'expires_at'      => $row->expires_at,
            'app_version'     => $app_version,
        ], 200);
    }

    /* ----------------- /start-trial ----------------- */

    public function handle_start_trial(WP_REST_Request $request) {
        global $wpdb;
        $table_act = $wpdb->prefix . self::TABLE_ACTIVATIONS;

        $params = $request->get_json_params();
        $device_id   = isset($params['device_id']) ? sanitize_text_field($params['device_id']) : '';
        $app_version = isset($params['app_version']) ? sanitize_text_field($params['app_version']) : '';

        if ($device_id === '') {
            return new WP_REST_Response([
                'status'  => 'error',
                'message' => 'Missing device_id.',
            ], 400);
        }

        $ip         = $this->get_client_ip();
        $ip_prefix  = $this->get_ip_prefix($ip);
        $user_agent = isset($_SERVER['HTTP_USER_AGENT']) ? substr(wp_unslash($_SERVER['HTTP_USER_AGENT']), 0, 255) : '';

        $device_hash = $this->make_device_hash($device_id, $user_agent);
        $now         = $this->now();

        $row = $wpdb->get_row(
            $wpdb->prepare(
                "SELECT * FROM $table_act WHERE device_hash = %s ORDER BY id DESC LIMIT 1",
                $device_hash
            )
        );

        if ($row) {
            $plan   = $row->plan;
            $status = $row->status;

            if ($plan !== 'TRIAL') {
                return new WP_REST_Response([
                    'status'  => $status,
                    'plan'    => $plan,
                    'message' => 'You already have a license or trial linked to this device.',
                ], 200);
            }

            if ($row->trial_expires_at && $this->is_expired($row->trial_expires_at)) {
                return new WP_REST_Response([
                    'status'  => 'expired',
                    'plan'    => 'TRIAL',
                    'message' => 'Your trial has already expired on this device.',
                ], 200);
            }

            $features = $this->feature_flags_for_plan('TRIAL');

            return new WP_REST_Response([
                'status'          => 'active',
                'plan'            => 'TRIAL',
                'message'         => 'Trial already active on this device.',
                'license_key'     => $this->mask_license($row->license_key),
                'trial_expires_at'=> $row->trial_expires_at,
                'features'        => $features,
                'app_version'     => $app_version,
            ], 200);
        }

        // Soft anti-abuse by IP prefix
        $max_trials_per_prefix = 3;
        $trial_count = (int) $wpdb->get_var(
            $wpdb->prepare(
                "SELECT COUNT(*) FROM $table_act
                 WHERE ip_prefix = %s AND plan = 'TRIAL'",
                $ip_prefix
            )
        );

        if ($trial_count >= $max_trials_per_prefix) {
            return new WP_REST_Response([
                'status'  => 'error',
                'plan'    => 'TRIAL_EXPIRED',
                'message' => 'Too many trial activations from your network. Please buy a license.',
            ], 403);
        }

        $trial_hours = 24 * 7;
        $trial_expires_at = $this->future_datetime($trial_hours);

        $wpdb->insert(
            $table_act,
            [
                'license_key'      => null,
                'device_id'        => $device_id,
                'device_hash'      => $device_hash,
                'plan'             => 'TRIAL',
                'status'           => 'active',
                'max_devices'      => 1,
                'used_devices'     => 1,
                'trial_expires_at' => $trial_expires_at,
                'expires_at'       => null,
                'ip_address'       => $ip,
                'ip_prefix'        => $ip_prefix,
                'user_agent'       => $user_agent,
                'last_check_at'    => $now,
                'app_version'      => $app_version,
                'created_at'       => $now,
                'updated_at'       => $now,
            ]
        );

        $features = $this->feature_flags_for_plan('TRIAL');

        return new WP_REST_Response([
            'status'          => 'active',
            'plan'            => 'TRIAL',
            'message'         => 'Trial started successfully.',
            'license_key'     => null,
            'trial_expires_at'=> $trial_expires_at,
            'features'        => $features,
            'app_version'     => $app_version,
        ], 200);
    }

    /* ----------------- /licenses (admin-only generator) ----------------- */

    public function handle_create_license(WP_REST_Request $request) {
        global $wpdb;
        $table_lic = $wpdb->prefix . self::TABLE_LICENSES;

        $params = $request->get_json_params();

        $plan        = isset($params['plan']) ? strtoupper(sanitize_text_field($params['plan'])) : 'PRO';
        $max_devices = isset($params['max_devices']) ? (int) $params['max_devices'] : 1;
        $expires_at  = isset($params['expires_at']) ? sanitize_text_field($params['expires_at']) : null;
        $notes       = isset($params['notes']) ? sanitize_textarea_field($params['notes']) : '';
        $license_key = isset($params['license_key']) ? sanitize_text_field($params['license_key']) : '';

        if (!in_array($plan, ['PRO', 'TRIAL', 'FREE'], true)) {
            $plan = 'PRO';
        }
        if ($max_devices < 1) {
            $max_devices = 1;
        }

        // If no license key provided, generate one
        if ($license_key === '') {
            do {
                $license_key = $this->generate_license_key(32);
                $exists = (int) $wpdb->get_var(
                    $wpdb->prepare(
                        "SELECT COUNT(*) FROM $table_lic WHERE license_key = %s",
                        $license_key
                    )
                );
            } while ($exists > 0);
        } else {
            // Ensure it doesn't already exist
            $exists = (int) $wpdb->get_var(
                $wpdb->prepare(
                    "SELECT COUNT(*) FROM $table_lic WHERE license_key = %s",
                    $license_key
                )
            );
            if ($exists > 0) {
                return new WP_REST_Response([
                    'status'  => 'error',
                    'message' => 'License key already exists.',
                ], 400);
            }
        }

        $now = $this->now();

        $wpdb->insert(
            $table_lic,
            [
                'license_key' => $license_key,
                'plan'        => $plan,
                'status'      => 'active',
                'max_devices' => $max_devices,
                'expires_at'  => $expires_at ? $expires_at : null,
                'notes'       => $notes,
                'created_at'  => $now,
                'updated_at'  => $now,
            ]
        );

        return new WP_REST_Response([
            'status'      => 'success',
            'message'     => 'License created.',
            'license_key' => $license_key,
            'plan'        => $plan,
            'max_devices' => $max_devices,
            'expires_at'  => $expires_at,
        ], 201);
    }
}

new Mayweb_License_API();
