<?php

/**
 * Copyright (c) 2019 BitPay
 **/
declare (strict_types=1);
namespace BitPayVendor\BitPaySDK\Functional;

use BitPayVendor\BitPaySDK\Exceptions\BitPayApiException;
use BitPayVendor\BitPaySDK\Exceptions\BitPayExceptionProvider;
use BitPayVendor\BitPaySDK\Exceptions\BitPayGenericException;
use BitPayVendor\BitPaySDK\Model\Currency;
use BitPayVendor\BitPaySDK\Model\Payout\Payout;
use BitPayVendor\BitPaySDK\Model\Payout\PayoutRecipient;
use BitPayVendor\BitPaySDK\Model\Payout\PayoutRecipients;
use BitPayVendor\BitPaySDK\Model\Payout\PayoutStatus;
class PayoutClientTest extends AbstractClientTestCase
{
    public function testPayoutRequests()
    {
        $currency = Currency::USD;
        $ledgerCurrency = Currency::USD;
        $amount = 10;
        $email = $this->getEmailFromFile(Config::FUNCTIONAL_TEST_PATH . \DIRECTORY_SEPARATOR . 'email.txt');
        $submitPayout = $this->submitPayout($currency, $ledgerCurrency, $amount);
        self::assertEquals($currency, $submitPayout->getCurrency());
        $payoutId = $submitPayout->getId();
        $payout = $this->client->getPayout($payoutId);
        self::assertEquals(10, $payout->getAmount());
        self::assertEquals($email, $payout->getNotificationEmail());
        $startDate = '2022-10-20T13:00:45.063Z';
        $endDate = '2023-01-01T13:00:45.063Z';
        $payouts = $this->client->getPayouts($startDate, $endDate);
        self::assertIsArray($payouts);
        self::assertCount(\count($payouts), $payouts);
        $requestPayoutNotification = $this->client->requestPayoutNotification($payoutId);
        self::assertTrue($requestPayoutNotification);
        $cancelledPayout = $this->client->cancelPayout($payoutId);
        self::assertTrue($cancelledPayout);
    }
    /**
     * @throws BitPayApiException
     */
    public function testPayoutGroupRequests() : void
    {
        $payout = new Payout();
        $payout->setAmount(10);
        $payout->setCurrency(Currency::USD);
        $payout->setLedgerCurrency(Currency::USD);
        $payout->setReference('payout_20210527');
        $payout->setNotificationEmail('merchant@email.com');
        $payout->setNotificationURL('https://yournotiticationURL.com/wed3sa0wx1rz5bg0bv97851eqx');
        $payout->setEmail($this->getEmailFromFile(Config::FUNCTIONAL_TEST_PATH . \DIRECTORY_SEPARATOR . 'email.txt'));
        $createGroupResponse = $this->client->createPayoutGroup([$payout]);
        self::assertCount(1, $createGroupResponse->getPayouts());
        self::assertEquals(PayoutStatus::NEW, $createGroupResponse->getPayouts()[0]->getStatus());
        $groupId = $createGroupResponse->getPayouts()[0]->getGroupId();
        $cancelGroupResponse = $this->client->cancelPayoutGroup($groupId);
        self::assertEquals(PayoutStatus::CANCELLED, $cancelGroupResponse->getPayouts()[0]->getStatus());
    }
    private function submitPayout(string $currency, string $ledgerCurrency, int $amount)
    {
        $email = $this->getEmailFromFile(Config::FUNCTIONAL_TEST_PATH . \DIRECTORY_SEPARATOR . 'email.txt');
        $payout = new Payout($amount, $currency, $ledgerCurrency);
        $recipientsList = [new PayoutRecipient($email, "recipient1", "https://yournotiticationURL.com/b3sarz5bg0wx01eq1bv9785amx")];
        $recipients = new PayoutRecipients($recipientsList);
        $payoutRecipients = $this->client->submitPayoutRecipients($recipients);
        $payoutRecipientId = $payoutRecipients[0]->getId();
        $payout->setRecipientId($payoutRecipientId);
        $payout->setNotificationURL("https://somenotiticationURL.com");
        $payout->setNotificationEmail($email);
        $payout->setReference("PHP functional tests " . \uniqid('', \true));
        $payout->setTransactions([]);
        return $this->client->submitPayout($payout);
    }
    /**
     * @throws BitPayGenericException
     */
    private function getEmailFromFile(string $path) : string
    {
        if (!\file_exists($path)) {
            BitPayExceptionProvider::throwGenericExceptionWithMessage("Please create email.txt with your email: " . $path);
        }
        return \file_get_contents($path);
    }
}
